# script.py (coding: utf-8)
#
# Copyright (c) 2009 by Iain Lamb <x+python@lamb.cc>
#
# Licensed to PSF under a Contributor Agreement.
# See http://www.python.org/2.6/license for details.

r"""Concise shell scripting

This module distills a collection of common shell script patterns into a few
terse definitions. Most of the objects and functions wrap lower-level calls to
modules in the standard Python library such as `optparse`, `os.path`, `shutil`,
and `subprocess`.

With just one `import` statement, you can provide your script with ready to go
command line parsing, convenient path operations, ala carte `--help`
documentation, quick writes to `stdout` / `stderr`, and easy invocation of shell
commands (including the ability to capture output).

requirements
------------

Python 3.0 or later.

usage
-----

	#!/usr/bin/env python
	from script import path, shell, err
	import script
	script.doc.purpose = \
		'How many flac files in ~/Music/flac belong to GENRE?'
	script.doc.args = 'GENRE'
	script.opts.preset('verbose')
	def main():
		if len(script.args) != 1:
			err('Please specify GENRE (or run with --help)')
			script.exit(1)
		genre = script.args[0].lower()
		count = 0
		root = path('~/Music/flac')
		err('scanning', root, when=script.opts.verbose)
		for parent, dirs, files in root.walk('*.flac'):
			for file in files:
				cmd = 'metaflac --show-tag=GENRE ' + path(parent/file).sh
				result = shell(cmd, stdout='PIPE').stdout
				if genre == result.lstrip('GENRE=').rstrip().lower():
					count += 1
		print('found {0} {1} files'.format(count, genre))
	
	if __name__ == '__main__': script.run(main)

This example could be run with `--help` or `--verbose`.

contents
--------

### `args`

This object starts out as an empty list, but after calling `run()` it contains any arguments passed via the command line.

### `doc`

If provided, the information in this object will be shown if the script is run with `--help`. Assign the `purpose` attribute to a string describing what your script does. Describe which arguments should be passed to your script by setting `args`.

	script.doc.args = "FILES"
	script.doc.purpose = "Save the World!"

### `err(*args, sep=' ', end='\n', when=True)`

If `when`, print `*args` to standard error, separated by `sep` followed by `end`.

### `exit(status=0)`

Identical to `sys.exit`

### `opts`

This object parses then stores the options passed from the command line. Prior to calling `run()`, you may call:

`opts.add()` : Adds whatever command-line options you wish to support.

The arguments match `optparse.OptionParser.add_option` with one exception: if the first argument does <em>not</em> start with `'-'` (and is a valid python identifier string), it will be expanded to a short option, a long option, and the value of the `dest` parameter. For example, calling `opts.add('foo', action='store_true')` is the same as calling:

	opts.add('-f', '--foo', dest='foo', action='store_true')

`opts.preset(*presets)` : This method provides a convenient way to add a few conventional command line options to the parser. `*presets` may be any of the following arguments:

`'clean'` invokes:

	opts.add('clean', action="store_true", help="remove previous output")

`'verbose'` invokes:

	opts.add('verbose', action="store_true", help="verbose output")
			
`'dry-run'` invokes:

	opts.add('-n', '--dry-run', dest="dry_run", action="store_true",
		help="preview script actions without doing them")
			
`'force'` invokes:

	opts.add('force', action="store_true",
		help="force action, ignoring warnings")

These presets merely add options the parser; your script must implement the appropriate response to them.

When `script.run()` is called, the `opts` object changes itself to contain whatever option values were parsed from the command line. It becomes the same kind of object typically returned from `optparse.OptionParser.parse_args()`.

### `out(*args, sep=' ', end='\n', when=True)`

If `when`, print `*args` to standard out, separated by `sep` followed by `end`.

### `parent`

Set to `path(sys.path[0])` when the `script` module is first imported. Unless `sys.path[0]` is altered before `import script`, this is the directory containing the script passed to the python interpreter.

### `path(name, expand=True, normalize=True)`

Instantiate and return a `Path` object, optionally expanding and normalizing the pathname prior to instantiation. Typically, this function is used more often than calling `Path` directly.

The `name` argument should be a pathname string.

If `expand` is `True`, the `name` string will be expanded using `os.path.expanduser(os.path.expandvars(name))`.

After any pathname expansion, if `normalize` is `True`, the `name` will be normalized via `os.path.normpath(name)`.

### `Path(name)`

A pathname or filename string, bundled with frequently used attributes and operations. Inspired by [PEP 355](http://www.python.org/dev/peps/pep-0355). 

Because *filenames are strings*, `Path` is a subclass of `str`.

The / operator may be used join paths into a new `Path` instance: `path('x') / 'y' == path('x/y')`

#### properties

- `abs` : `os.path.abspath`
- `name` : `os.path.basename`
- `parent` : `os.path.dirname`
- `ext` : `(os.path.splitext)[-1]`
- `no_ext` : `(os.path.splitext)[0]`
- `exists` : `os.path.exists`
- `mtime` : `os.path.getmtime`
- `is_file` : `os.path.isfile`
- `is_dir` : `os.path.isdir`
- `size` : `os.path.getsize`
- `sh` : shell escaped pathname
- `usize` : `'{size:3g}{unit}'`

All these are read-only `property` attributes. Except for `sh` and `usize`, each calls the function listed.

The `abs`, `name`, and `parent` properties are Path instances.

To just get the name of the file, without a path prefix or an extension, use `.name.no_ext`

The `sh` property computes an escaped path string suitable for evaluation in the bash shell. For example:

	path('( &!?*"@`;,\)').sh == '\(\ \&\!\?\*\"\@\`\;\,\\)'

The `usize` property computes a string indicating the size of the file in three digits or less followed by a unit suffix: (B)ytes, (K)ilobytes, (M)egabytes, (G)igabytes, (T)erabytes, or (P)etabytes.

#### methods

- `list(pattern=None)` : iterator over `os.listdir` yielding self/filename
- `walk(pattern=None)` : iterator over `os.walk`
- `touch(mode=0o644)` : unix touch command
- `mkdir(mode=0o755)` : `os.makedirs` if not (`.exists` and `.is_dir`)
- `open_utf8(mode='r')` : call `open()` with `encoding='utf8'`
- `match` : `fnmatch.fnmatch`
- `cd`, `chdir` : `os.chdir`
- `chown` : `os.chown`
- `remove` : `os.remove`
- `rename` : `os.rename`
- `rmdir` : `os.rmdir`
- `rmtree` : `shutil.rmtree`
- `copy` : `shutil.copy`
- `older(other)` : `self.mtime < path(other).mtime`
- `newer(other)` : `self.mtime > path(other).mtime`

Note that `list`, `walk`, and `mkdir` return or yield Path instances.

The `list` function iterates over each stand alone filename returned from `os.listdir` and returns self/filename (e.g. the filename prepended with path information)

The `mkdir` method does not throw an `OSError` if the directory already exists.

Items yielded by `list` may be filtered to match `pattern`. For example, `list('*.py')` yields only Path instances with `.ext == '.py'`.

The `walk` method yields `(dirname,dirs,files)`. The items in `files` may also be filtered to match `pattern`. For example, `walk('*.py')` yields a `files` list containing only Path instances with `.ext == '.py'`.

### `paths(*args)`

Returns an iterator yielding `path(p)` for each item `glob.iglob(arg)` in args. Each argument may be a `str` or an iterator, in which case the glob is applied to every item inside it.

### `run(main)`

Once `opts` and `doc` are configured to your liking, call this method to parse the command line and call `main()`. Exit quietly, with return code of 1, if the user presses control-c.

### `shell(command, stdin=None, stdout=None, stderr=None, failcode='non-zero')`

Runs a single `command` string in the shell as a subprocess.

The `stdin`, `stdout`, and `stderr` parameters control the standard input, output, and error file handles of the subprocess. By default, each one defaults to `None` to indicate that the subprocess should use the `stdin`, `stdout`, or `stderr` of the `sys` module. Each may be set to an existing file object or file descriptor (a positive integer).

You may redirect the standard error of the subprocess into the same file handle as its standard output by setting `stderr` to `'STDOUT'`.

To capture the standard output or standard error of the subprocess into a `bytes` value in memory, set either `stdout` or `stderr` to `'PIPE'`. Avoid this setting if the subprocess is going to return an especially large or unlimited amount of data.

Once the subprocess completes, a `script.CommandResult` object is returned with the following attributes:

- `stdout_bytes`, `stderr_bytes` : `None` or a `bytes` value, see below.
- `stdout`, `stderr` : `None` or a `str` value, see below.
- `returncode` : the return code of the subprocess.

If the `returncode` attribute matches the `failcode` parameter passed into the call to `shell()`, a `script.CommandFailed` exception will be raised. The `failcode` parameter may be set to `None`, a specific return code, or `'non-zero'` to match any non-zero return code.

If the `stdout` or `stderr` parameter passed into the call to `shell()` was set to `'PIPE'`, the `stdout_bytes` or `stderr_bytes` attribute will be a `bytes` value containing all the standard output or error produced by the subprocess.

When the `stdout_bytes` or `stderr_bytes` property contains a `bytes` value, a reference to the `stdout` or `stderr` property will have the same effect as calling `stdout_bytes.decode()` or `stderr_bytes.decode()`. 

"""

import os, subprocess, fnmatch, shutil, sys, glob, re
from optparse import OptionParser

__version__ = '1.4.1'

# public symbols
__all__ = [ 'args', 'doc', 'err', 'exit', 'opts', 'out', 'parent', 'path', 'Path', 'paths', 'run', 'shell']

args = list()
class Doc:

	def __init__(self):
		self.args = self.purpose = ''
	

doc = Doc()
def err(*args, sep=' ', end='\n', when=True):
	if when: print(*args, sep=sep, end=end, file=sys.stderr)

def exit(status=0): sys.exit(status)

def __preset(*presets):
	for preset in presets:
		# backwards compatibility:
		def warn(flag, value):
			nonlocal preset
			err('WARNING: the "{0}" preset is deprecated. Please use "{1}" instead'.format(flag, value))
			preset = value
		
		if preset in ['-c', '--clean']: warn(preset, 'clean')
		elif preset in ['-v', '--verbose']: warn(preset, 'verbose')
		elif preset in ['-n', '--dry-run']: warn(preset, 'dry-run')
		elif preset in ['-f', '--force']: warn(preset, 'force')
		
		if preset == 'clean': opts.add('clean', action="store_true", help="remove previous output")
		elif preset == 'verbose': opts.add('verbose', action="store_true", help="verbose output")
		elif preset == 'force': opts.add('force', action="store_true", help="ignore warnings, force action")
		elif preset == 'dry-run' or preset == 'dry_run': opts.add('-n', '--dry-run', dest="dry_run", action="store_true", help="preview actions without doing them")
		else: raise ValueError('Unknown preset: {0}'.format(preset))

def __parse():
	return opts.parser.parse_args()

def __add(*args, **kwargs):
	if len(args):
		first = args[0]
		if first and first[0] != '-' and re.match(r'[A-Za-z_-][A-Za-z0-9_-]*$', first):
			args = ['-{0}'.format(first[0]), '--{0}'.format(first)]
			kwargs['dest'] = first.replace('-', '_')
	opts.parser.add_option(*args,**kwargs)

class Opts:
	def __init__(self, preset, parse, add):
		self.parser = OptionParser()
		self.add = add
		self.preset = preset
		self.parse = parse
	

opts = Opts(__preset,__parse,__add)

def out(*args, sep=' ', end='\n', when=True):
	if when: print(*args, sep=sep, end=end, file=sys.stdout)

class Path(str):
	def __truediv__(self, name): return path(os.path.join(self, name))
	
	abs = property(lambda self : path(os.path.abspath(self)))
	name = property(lambda self : path(os.path.basename(self)))
	parent = property(lambda self : path(os.path.dirname(self)))
	exists = property(lambda self : os.path.exists(self))
	ext = property(lambda self : os.path.splitext(self)[-1])
	no_ext = property(lambda self : os.path.splitext(self)[0])
	mtime = property(lambda self : os.path.getmtime(self))
	is_file = property(lambda self : os.path.isfile(self))
	is_dir = property(lambda self : os.path.isdir(self))
	size = property(lambda self : os.path.getsize(self))
	def __usize(self):
		size = self.size
		for label in ['B', 'K', 'M', 'G', 'T', 'P']:
			if size < 1024: break
			else: size /= 1024.0
		return '{0:.3g}{1}'.format(size, label)
	
	usize = property(__usize)
	def __sh(self):
		for c in ['\\', ' ', '(', ')', '&', '!', '?', '*', "'", '"', '@', '`', ';', ',']:
			self = self.replace(c, '\\' + c)
		return str(self)
	
	sh = property(__sh)
	def list(self, pattern=None):
		for name in os.listdir(self):
			if pattern != None and not fnmatch.fnmatch(name, pattern): continue
			yield self/name
	
	def walk(self, pattern=None, topdown=True, onerror=None, followlinks=False):
		for (dirname, dirs, files) in os.walk(self, topdown, onerror, followlinks):
			if pattern != None:	files = fnmatch.filter(files, pattern)
			for list in dirs, files:
				for index, name in enumerate(list): list[index] = path(name)
			yield (path(dirname), dirs, files)
	
	def mkdir(self, mode=0o755):
		if not (self.exists and self.is_dir): os.makedirs(self, mode)
		return self
		
	def open_utf8(self, mode='r'): return open(self, mode, encoding='utf8')
	
	def touch(self, mode=0o644):
		fd = os.open(self, os.O_WRONLY | os.O_CREAT)
		os.close(fd)
		os.chmod(self, mode)
		os.utime(self, None)
	
	def match(self, pattern): return fnmatch.fnmatch(self, pattern)
	
	def chdir(self): os.chdir(self)
	cd = chdir
	def chown(self, uid=-1, gid=-1): os.chown(self, uid, gid)
	
	def remove(self): os.remove(self)
	
	def rename(self, dst): os.rename(self, dst)
	
	def rmdir(self): os.rmdir(self)
	
	def rmtree(self, ignore_errors=False, onerror=None):
		shutil.rmtree(self, ignore_errors, onerror)
	
	def copy(self, dst): shutil.copy(self, dst)
	
	def older(self, other): return self.mtime < path(other).mtime
	
	def newer(self, other): return self.mtime > path(other).mtime
	

def path(name, expand=True, normalize=True):
	if expand == True: name = os.path.expanduser(os.path.expandvars(name))
	if normalize == True: name = os.path.normpath(name)
	return Path(name)

def paths(*args):
	for arg in args:
		if issubclass(arg.__class__, str): arg = [arg]
		for pattern in arg:
			found = False
			for pathname in glob.iglob(pattern):
				found = True
				yield path(pathname)
			if not found:
				if '*' in pattern: continue
				# the file does not exist, yet
				yield path(pattern)

def run(main):
	# parse the command line
	usage = "%prog [options]"
	if doc.args != '':
		usage += " {0}".format(doc.args)
	if doc.purpose != '':
		usage += '\n\nPurpose: {0}'.format(doc.purpose)
	opts.parser.usage = usage
	(options, arguments) = opts.parse()
	for attr in ('parser','add','preset','parse'): delattr(opts,attr)
	for key in options.__dict__: opts.__dict__[key] = options.__dict__[key]
	args[:] = arguments
	# run the main funciton of the script
	try:
		main()
	except KeyboardInterrupt:
		exit(1)

parent = path(sys.path[0])

class CommandFailed(Exception): pass

class CommandResult:
	def __init__(self, stdout_bytes=None, stderr_bytes=None, returncode=None):
		self.stdout_bytes = stdout_bytes
		self.stderr_bytes = stderr_bytes
		self.returncode = returncode
		
	stdout = property(lambda self : self.stdout_bytes.decode() if self.stdout_bytes != None else None)
	stderr = property(lambda self : self.stderr_bytes.decode() if self.stderr_bytes != None else None)
	

def shell(command, stdin=None, stdout=None, stderr=None, failcode='non-zero'):
		
	def normalize(type):
		if type == 'PIPE': return subprocess.PIPE
		elif type == 'STDOUT': return subprocess.STDOUT
		else: return type
	
	process = subprocess.Popen(command,
		stdin=normalize(stdin),
		stdout=normalize(stdout),
		stderr=normalize(stderr),
		shell=True)
	
	stdout_bytes, stderr_bytes = process.communicate()
	if failcode == process.returncode or (failcode == 'non-zero' and process.returncode != 0):
		raise CommandFailed("Command returned failure status: ({0})".format(process.returncode))
	else:
		return CommandResult(stdout_bytes, stderr_bytes, process.returncode)
	

